# Web Deployment Architecture

## Overview
The Provider Search Flutter app is deployed as a static web application using a reverse proxy architecture with nginx.

## Architecture

### Network Flow
```
Internet → Router → Main Nginx (192.168.1.2) → Container (192.168.1.22:80)
```

### Components

1. **Router**: Forwards all web traffic to main nginx server at 192.168.1.2
2. **Main Nginx Server (192.168.1.2)**: 
   - Handles all SSL/HTTPS termination using Let's Encrypt certificates
   - Routes traffic based on hostname (app.dataintegrities.com, downloads.dataintegrities.com, etc.)
   - Forwards requests to appropriate internal servers
3. **Container (192.168.1.22)**: 
   - Proxmox Linux container 
   - Runs simple nginx serving static files
   - No SSL needed (handled by main nginx)

### Main Nginx Configuration
Located at `/etc/nginx/sites-available/di-app` on 192.168.1.2:
```nginx
server {
  set $forward_scheme http;
  set $server         "192.168.1.22";  # Container IP
  set $port           80;               # HTTP only to container

  listen 80;
  listen 443 ssl;
  server_name app.dataintegrities.com;
  
  # SSL handled here with Let's Encrypt
  ssl_certificate /etc/letsencrypt/live/dataintegrities.com/fullchain.pem;
  ssl_certificate_key /etc/letsencrypt/live/dataintegrities.com/privkey.pem;
  
  location / {
    # Forwards to container at 192.168.1.22:80
    include /etc/nginx/conf.d/include/proxy.conf;
  }
}
```

## Container Setup

### Current Implementation (Node.js - TO BE REPLACED)
- Files deployed to: `/home/jeffk/dev/provider-search-flutter/`
- Node.js app server serves from: `../build/web/` (relative to app-server directory)
- Managed by PM2 process ID 1
- Handles both HTTP (80) and HTTPS (443) - but HTTPS not needed due to proxy

### Target Implementation (nginx)
- Web files: `~/www/` (Flutter build output)
- Simple nginx config serving static files on port 80
- No SSL certificates needed (handled by main nginx)
- No PM2 or Node.js dependencies

### Directory Structure (Target)
```
/home/jeffk/
├── www/                              # Web files (nginx document root)
│   ├── index.html
│   ├── main.dart.js
│   ├── flutter.js
│   └── assets/
└── dev/
    └── provider-search-flutter/      # Build workspace (GitHub Actions)
        └── build/web/                # Flutter build output (source for ~/www/)
```

## Deployment Process

### Current Process (Node.js)
1. GitHub Actions builds Flutter web app
2. Copies entire project to `/home/jeffk/dev/provider-search-flutter/`
3. Installs Node.js dependencies in `app-server/`
4. Stops PM2 process 1
5. Restarts PM2 process 1
6. Node.js serves static files from `../build/web/`

### Target Process (nginx)
1. GitHub Actions builds Flutter web app
2. Copies Flutter web files from `./build/web/*` to `~/www/`
3. nginx automatically serves updated files (no restart needed)

## Benefits of nginx Approach
- **Simpler**: No Node.js, PM2, or npm dependencies
- **Faster**: Direct static file serving vs Express.js overhead
- **More reliable**: nginx is battle-tested for static files
- **Standard**: Industry standard approach for static web apps
- **Easier troubleshooting**: Standard nginx logs and configuration

## Installation Commands
```bash
# Install nginx via Homebrew (on Linux container)
brew install nginx

# Create directories
mkdir -p ~/www

# Copy current web files for testing
cp -r /home/jeffk/dev/provider-search-flutter/build/web/* ~/www/

# Configure nginx to serve from ~/www on port 80
```

## Current SVG Asset Issue (July 2025)

### Problem
Flutter web SVG assets are returning 403 Forbidden errors because nginx is proxying ALL requests (including static assets) to the Node.js backend instead of serving them directly.

### Root Cause
The current `/etc/nginx/include/nodejs-proxy` configuration proxies everything to Node.js:
```nginx
location / {
    proxy_pass http://127.0.0.1:8000;
    # ... proxy settings
}
```

### Solution
Modify `/etc/nginx/include/nodejs-proxy` to serve static assets directly:
```nginx
# Serve Flutter web assets directly from nginx
location /assets/ {
    root /home/jeffk/www;
    expires 1y;
    add_header Cache-Control "public, immutable";
    try_files $uri $uri/ =404;
}

# Serve other static files directly 
location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)$ {
    root /home/jeffk/www;
    expires 1y;
    add_header Cache-Control "public, immutable";
}

# Proxy everything else to Node.js
location / {
    proxy_pass http://127.0.0.1:8000;
    # ... existing proxy settings
}
```

### Server Details
- **Web Files Location**: `/home/jeffk/www/` (deployed by GitHub Actions)
- **Nginx Config**: `/etc/nginx/include/nodejs-proxy`
- **Service Management**: Use `brew services` instead of `systemctl`
- **Test Config**: `nginx -t` (if available via brew)
- **Reload**: `brew services reload nginx`

## Notes
- Container runs on Proxmox virtualization platform
- Uses Homebrew package manager on Linux (not systemctl)
- Main nginx server handles all SSL complexity
- Container nginx only needs HTTP on port 80
- Flutter web apps are completely static (no server-side processing needed)
- SVG MIME type is already configured in `/etc/nginx/mime.types`